<?php

namespace App\Http\Controllers;

use App\Models\Dispositivo;
use Illuminate\Auth\Access\AuthorizationException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class DispositivoController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth:api');
    }
    public function crearDispositivo(Request $request)
    {
        try {
            $this->authorize('registrar-dispositivo');
            $validator = Validator::make($request->all(), [
                'entidad_id' => 'required|exists:entidades_comerciales,id',
                'ultima_conexion' => 'required',
                'nombre_dispositivo' => 'required',
                'ubicacion' => 'required',
                'observaciones_diagnostico' => 'required',
                'estado' => 'required',
                'estado_diagnostico' => 'required',
                'version_apk_instalada' => 'required',
                'ip_publica' => 'required',
                'codigo_remoto' => 'required',
                'herramienta_remoto' => 'required',
                'token_push_FCM' => 'required',
            ]);
            if ($validator->fails()) {
                return response()->json([
                    'status' => 422,
                    'message' => $validator->errors(),
                ], 422);
            } else {
                $dispositivo = Dispositivo::create([
                    'entidad_id' => $request->entidad_id,
                    'ultima_conexion' => $request->ultima_conexion,
                    'nombre_dispositivo' => $request->nombre_dispositivo,
                    'ubicacion' => json_encode($request->ubicacion),
                    'observaciones_diagnostico' => $request->observaciones_diagnostico,
                    'estado' => $request->estado,
                    'estado_diagnostico' => $request->estado_diagnostico,
                    'version_apk_instalada' => $request->version_apk_instalada,
                    'ip_publica' => $request->ip_publica,
                    'codigo_remoto' => $request->codigo_remoto,
                    'herramienta_remoto' => $request->herramienta_remoto,
                    'token_push_FCM' => $request->token_push_FCM,
                ]);
                return response()->json([
                    'status' => 201,
                    'message' => 'Dispositivo creado correctamente',
                    'data' => $dispositivo,
                ], 201);
            }
        } catch (AuthorizationException $th) {
            return response()->json([
                'status' => $th->getCode(),
                'message' => 'No tiene permisos para crear dipositivos',
                'data' => $th->getMessage(),
            ], 401);
        } catch (\Exception $e) {
            return response()->json([
                'status' => $e->getCode(),
                'message' => 'Error al crear el dispositivo',
                'data' => $e->getMessage(),
            ], 400);
        }
    }
    public function actualizarDispositivo(Request $request, $id)
    {
        try {
            $this->authorize('registrar-dispositivo');

            $dispositivo = Dispositivo::findOrFail($id);

            $dispositivo->fill($request->all());
            if ($request->has('ubicacion')) {
                $dispositivo->ubicacion = json_encode($request->ubicacion);
            }

            $dispositivo->save();

            return response()->json([
                'status' => 200,
                'message' => 'Dispositivo actualizado correctamente',
                'data' => $dispositivo,
            ], 200);
        } catch (AuthorizationException $th) {
            return response()->json([
                'status' => $th->getCode(),
                'message' => 'No tiene permisos para actualizar dispositivos',
                'data' => $th->getMessage(),
            ], 401);
        } catch (\Exception $e) {
            return response()->json([
                'status' => $e->getCode(),
                'message' => 'Error al actualizar el dispositivo',
                'data' => $e->getMessage(),
            ], 400);
        }
    }
    public function listarDispositivo(Request $request)
    {
        try {
            $request->merge([
                'page' => $request->input('page', 0) + 1
            ]);
            $dispositivo = Dispositivo::orderBy('updated_at', 'desc')
                ->paginate($request->query('per_page') ?? 10);
            return ($dispositivo != null) ?
                response()->json([
                    'status' => 200,
                    'message' => 'Dispositivos listados correctamente.',
                    'data' => $dispositivo
                ], 200) :
                response()->json([
                    'status' => 200,
                    'message' => 'No hay registro de Dispositivos.',
                    'data' => null
                ], 200);
        } catch (\Throwable $th) {
            return response()->json([
                'status' => $th->getCode(),
                'message' => 'Ocurrio un error!.',
                'data' => $th->getMessage()
            ], 400);
        }
    }
    public function consultarDispositivoPorEntidad($entidadId)
    {
        try {
            $this->authorize('registrar-dispositivo');

            $dispositivo = Dispositivo::where('entidad_id', $entidadId)->first();

            if ($dispositivo) {
                return response()->json([
                    'status' => 200,
                    'message' => 'Dispositivo encontrado',
                    'data' => $dispositivo,
                ], 200);
            } else {
                return response()->json([
                    'status' => 404,
                    'message' => 'No se encontró ningún dispositivo para la entidad especificada',
                ], 404);
            }
        } catch (AuthorizationException $th) {
            return response()->json([
                'status' => $th->getCode(),
                'message' => 'No tiene permisos para consultar dispositivos',
                'data' => $th->getMessage(),
            ], 401);
        } catch (\Exception $e) {
            return response()->json([
                'status' => $e->getCode(),
                'message' => 'Error al consultar el dispositivo',
                'data' => $e->getMessage(),
            ], 400);
        }
    }

    // public function verApiCoorporativo($id)
    // {
    //     try {
    //         // $this->authorize('ver-api-coorporativo');
    //         $apiCoorporativo = $this->apiCoorporativo->getById($id);
    //         return ($apiCoorporativo) ? response()->json([
    //             'status' => 200,
    //             'message' => 'Api encontrada',
    //             'data' => $apiCoorporativo,
    //         ], 200) : response()->json([
    //             'status' => 200,
    //             'message' => 'No se encontro la api',
    //             'data' => null,
    //         ], 200);
    //     } catch (AuthorizationException $th) {
    //         return response()->json([
    //             'status' => $th->getCode(),
    //             'message' => 'No tiene permisos para ver la api',
    //             'data' => $th->getMessage(),
    //         ], 401);
    //     } catch (\Exception $e) {
    //         return response()->json([
    //             'status' => $e->getCode(),
    //             'message' => 'Error al ver la api',
    //             'data' => $e->getMessage(),
    //         ], 400);
    //     }
    // }
    // public function verApiCoorporativoEntidad($id)
    // {
    //     try {
    //         // $this->authorize('ver-api-coorporativo-entidad');
    //         $apiCoorporativo = $this->apiCoorporativo->getByIdEntidad($id);
    //         return ($apiCoorporativo) ? response()->json([
    //             'status' => 200,
    //             'message' => 'Api encontrada',
    //             'data' => $apiCoorporativo,
    //         ], 200) : response()->json([
    //             'status' => 200,
    //             'message' => 'No se encontro la api',
    //             'data' => null,
    //         ], 200);
    //     } catch (AuthorizationException $th) {
    //         return response()->json([
    //             'status' => $th->getCode(),
    //             'message' => 'No tiene permisos!',
    //             'data' => $th->getMessage(),
    //         ], 401);
    //     } catch (\Exception $e) {
    //         return response()->json([
    //             'status' => $e->getCode(),
    //             'message' => 'Ocurrio un error!',
    //             'data' => $e->getMessage(),
    //         ], 400);
    //     }
    // }

    // public function actualizarClaveApiCoorporativo($id){
    //     try {
    //         // $this->authorize('actualizarClaveApi');
    //         $apiCoorporativo = $this->apiCoorporativo->updateClaveById($id);
    //         if($apiCoorporativo!=null){
    //             return response()->json([
    //                 'status' => 200,
    //                 'message' => 'Clave actualizada correctamente',
    //                 'data' => $apiCoorporativo,
    //             ], 200);
    //         }
    //         else{
    //             return response()->json([
    //                 'status' => 200,
    //                 'message' => 'No se encontro la api',
    //                 'data' => null,
    //             ], 200);
    //         }
    //     } catch (AuthorizationException $th) {
    //         return response()->json([
    //             'status' => $th->getCode(),
    //             'message' => 'No tiene permisos para actualizar la clave de la api',
    //             'data' => $th->getMessage(),
    //         ], 401);
    //     }catch (\Exception $e) {
    //         return response()->json([
    //             'status' => $e->getCode(),
    //             'message' => 'Error al actualizar la clave de la api',
    //             'data' => $e->getMessage(),
    //         ], 400);
    //     }
    // }

    // public function enviarCredenciales(Request $request)
    // {
    //     try {
    //         // $this->authorize('enviar-credenciales-api-coorporativo');
    //         $validator = Validator::make($request->all(), [
    //             'entidad_id' => 'required|exists:entidades_comerciales,id',

    //         ]);
    //         if ($validator->fails()) {
    //             return response()->json([
    //                 'status' => 422,
    //                 'message' => 'Error al validar datos',
    //                 'data' => $validator->errors(),
    //             ], 422);
    //         } else {
    //             $entidad = EntidadComercial::find($request->entidad_id)->load('api_coorporativo');
    //             $email = $this->apiCoorporativo->sendEmail($entidad, $request->correos);
    //             if ($email) {
    //                 return response()->json([
    //                     'status' => 200,
    //                     'message' => 'Credenciales enviadas correctamente',
    //                     'data' => $email,
    //                 ], 200);
    //             } else {
    //                 return response()->json([
    //                     'status' => 200,
    //                     'message' => 'No se encontro la api',
    //                     'data' => null,
    //                 ], 200);
    //             }
    //         }
    //     } catch (AuthorizationException $th) {
    //         return response()->json([
    //             'status' => $th->getCode(),
    //             'message' => 'No tiene permisos!',
    //             'data' => $th->getMessage(),
    //         ], 401);
    //     } catch (\Exception $e) {
    //         return response()->json([
    //             'status' => $e->getCode(),
    //             'message' => 'Error al enviar las credenciales',
    //             'data' => $e->getMessage(),
    //         ], 400);
    //     }
    // }
}
