<?php

namespace App\Http\Controllers;

use App\Models\DiagnosticoDispositivo;
use App\Models\LogDispositivo;
use Illuminate\Auth\Access\AuthorizationException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;

class LogDispositivoAuthController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth:api');
    }

    public function listarLogDispositivos(Request $request)
    {
        try {
            $this->authorize('registrar-dispositivo');
            $request->merge([
                'page' => $request->input('page', 0) + 1
            ]);
            $listLogs = LogDispositivo::with('dispositivo')
                ->orderBy('updated_at', 'desc')
                ->paginate($request->query('per_page') ?? 10);
            return ($listLogs != null) ?
                response()->json([
                    'status' => 200,
                    'message' => 'Disgnosticos en dispositivos listados correctamente.',
                    'data' => $listLogs
                ], 200) :
                response()->json([
                    'status' => 200,
                    'message' => 'No hay registro de Disgnosticos en dispositivos.',
                    'data' => null
                ], 200);
        } catch (\Throwable $th) {
            return response()->json([
                'status' => $th->getCode(),
                'message' => 'Ocurrio un error!.',
                'data' => $th->getMessage()
            ], 400);
        }
    }

    public function listarLogsDispositivo(Request $request, $dispositivoId)
    {
        try {
            $this->authorize('registrar-dispositivo');
            $request->merge([
                'page' => $request->input('page', 0) + 1
            ]);
            $validator = Validator::make($request->all(), [
                'fecha_desde' => 'required',
                'fecha_hasta' => 'required',
            ]);
            if ($validator->fails()) {
                return response()->json([
                    'status' => 422,
                    'message' => $validator->errors(),
                ], 422);
            } else {
                $fechaDesde = $request->input('fecha_desde');
                $fechaHasta = $request->input('fecha_hasta');
                // dd($fechaDesde, $fechaHasta);
                $query = LogDispositivo::with('dispositivo')
                    ->where('dispositivo_id', $dispositivoId)
                    ->whereBetween('updated_at', [$fechaDesde, $fechaHasta . ' 23:59:59'])
                    ->orderBy('updated_at', 'desc');

                $listLogs = $query->paginate($request->query('per_page') ?? 10);
                // dd($listLogs);
                return ($listLogs->count() > 0) ?
                    response()->json([
                        'status' => 200,
                        'message' => 'Logs del dispositivo listados correctamente.',
                        'data' => $listLogs
                    ], 200) :
                    response()->json([
                        'status' => 200,
                        'message' => 'No hay registro de Logs en el dispositivo.',
                        'data' => null
                    ], 200);
            }
        } catch (\Throwable $th) {
            return response()->json([
                'status' => $th->getCode(),
                'message' => 'Ocurrió un error.',
                'data' => $th->getMessage()
            ], 400);
        }
    }

    public function listarLogsDiaDispositivo(Request $request, $dispositivoId)
    {
        try {
            $this->authorize('registrar-dispositivo');
            $request->merge([
                'page' => $request->input('page', 0) + 1
            ]);

            $fechaActual = Carbon::now()->toDateString();

            $query = LogDispositivo::with('dispositivo')
                ->where('dispositivo_id', $dispositivoId)
                ->whereDate('updated_at', $fechaActual)
                ->orderBy('updated_at', 'desc');

            $listLogs = $query->paginate($request->query('per_page') ?? 10);

            return ($listLogs->count() > 0) ?
                response()->json([
                    'status' => 200,
                    'message' => 'Logs del dispositivo listados correctamente.',
                    'data' => $listLogs
                ], 200) :
                response()->json([
                    'status' => 200,
                    'message' => 'No hay registro de Logs en el dispositivo.',
                    'data' => null
                ], 200);
        } catch (\Throwable $th) {
            return response()->json([
                'status' => $th->getCode(),
                'message' => 'Ocurrió un error.',
                'data' => $th->getMessage()
            ], 400);
        }
    }
}
