<?php

namespace App\Http\Controllers;

use App\Mail\ResetPasswordMail;
use App\Models\Usuario;
use App\Repositories\AccesoRepository;
use Carbon\Carbon;
use Illuminate\Auth\Events\PasswordReset;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Facades\Validator;
use PHPOpenSourceSaver\JWTAuth\Facades\JWTAuth;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\dd;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Exception;

class LoginController extends Controller
{
    protected $guard = 'usuarios';
    private AccesoRepository $acceso;
    public function __construct(AccesoRepository $acceso)
    {
        $this->acceso = $acceso;
    }
    public function login(Request $request)
    {
        try {
            $request->validate([
                'correo' => 'required|string|email',
                'password' => 'required|string',
            ]);
            $credentials = $request->only('correo', 'password');
            $token = JWTAuth::attempt($credentials);
            if (!$token) {
                return response()->json([
                    'status' => 401,
                    'message' => 'Datos incorrectos',
                    'data' => null
                ], Response::HTTP_UNAUTHORIZED);
            } else {
                $user = JWTAuth::user();
                $this->acceso->createAcceso([
                    'usuario_id' => $user->id,
                    'fecha' => Carbon::now(),
                    'ip_acceso' => $request->ip()
                ]);


                return response()->json([
                    'status' => 200,
                    'authorisation' => [
                        'data' => $user->load('tipo_usuario', 'entidad_id'),
                        'token' => $token,
                        'type' => 'bearer',
                        'expires_in' => (auth('api')->factory()->getTTL()) / 60 . ' horas'
                    ]
                ]);
            }
        } catch (\Throwable $th) {
            return response()->json([
                'status' => $th->getCode(),
                'message' => 'Error al intentar iniciar sesion.',
                'data' => $th->getMessage()

            ], 400);
        }
    }

    public function sendResetLinkEmail(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), ['correo' => 'required|email|exists:usuarios,correo']);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 422,
                    'message' => 'Error al validar los datos de entrada.',
                    'data' => $validator->errors()
                ], 422);
            } else {
                $email = $request->input('correo');

                $token = Str::random(64);

                DB::table('password_resets')->insert([
                    'email' => $email,
                    'token' => $token,
                    'created_at' => now()
                ]);

                $passwordResetObject = DB::table('password_resets')
                    ->where('email', $email)
                    ->where('token', $token)
                    ->first();

                if ($passwordResetObject) {
                    Mail::to($email)->send(new ResetPasswordMail($passwordResetObject));
                    return response()->json(['message' => 'Se ha enviado un enlace de restablecimiento de contraseña a tu correo electrónico.'], 200);
                } else {
                    return response()->json(['error' => 'No se pudo generar el enlace de restablecimiento de contraseña. Por favor, intenta nuevamente.'], 500);
                }
            }
        } catch (Exception $e) {
            return response()->json([
                'status' => $e->getCode(),
                'message' => 'Ocurrio un error!.',
                'data' => $e->getMessage()
            ], 400);
        }
    }


    protected function cambiarClave(Request $request)
    {

        $input = $request->only('correo', 'token', 'password', 'password_confirmation');
        $validator = Validator::make($input, [
            'token' => 'required',
            'correo' => 'required|email',
            'password' => 'required|confirmed|min:8',
        ]);

        if ($validator->fails()) {
            return response([
                'status' => 422,
                'message' => 'Ocurrió un error al validar los datos.',
                'data' => $validator->errors()
            ], 422);
        }

        $tokenData = DB::table('password_resets')
            ->where('email', $input['correo'])
            ->where('token', $input['token'])
            ->first();

        if (!$tokenData) {
            return response([
                'status' => 422,
                'message' => 'El token de recuperación de contraseña no es válido.',
                'data' => []
            ], 422);
        }

        // Verificación exitosa del token, proceder a cambiar la contraseña
        $user = Usuario::where('correo', $input['correo'])->first();
        $user->password = Hash::make($input['password']);
        $user->save();

        // Eliminar el registro del token de recuperación de contraseña
        DB::table('password_resets')
            ->where('email', $input['correo'])
            ->delete();

        // Acciones adicionales después de actualizar la contraseña
        // ...

        return response([
            'status' => 200,
            'message' => 'La contraseña se ha cambiado correctamente.',
            'data' => []
        ], 200);
    }

    public function getClientIPaddress(Request $request)
    {
        $clientIp = $request->ip();
        return $clientIp;
    }
}
