<?php

namespace App\Http\Controllers;

use App\Models\SolicitudPrevia;
use App\Models\TipoFirma;
use Carbon\Carbon;
use Exception;
use Illuminate\Auth\Access\AuthorizationException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Mail;
use App\Mail\EmailSolicitudPrevia;
use App\Models\EntidadComercial;

class SolicitudPreviaAuthController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth:api');
    }
    public function crearNuevaSolicitudAuth(Request $request)
    {
        try {
            // $this->authorize('registrar-entidadesComerciales');
            $validator = Validator::make($request->all(), [
                'email' => 'required',
                'identificacion' => 'required',
                'telefono' => 'required',
                'requiere_facturacion' => 'required',
                'numero_facturas' => 'required',
                'requiere_firma' => 'required',
                'precio' => 'required',
                'tipo_persona' => 'required',
                'entidad_id' => 'required|exists:entidades_comerciales,id',
                'tipo_firma_id' => 'required|exists:tipos_firmas,id',
                'usuario_corp' => 'required|exists:usuarios,id',
            ]);
            if ($validator->fails()) {
                return response()->json([
                    'status' => 422,
                    'message' => 'Error al validar los datos de entrada.',
                    'data' => $validator->errors()
                ], 422);
            } else {
                $tipo_firma = TipoFirma::find($request->tipo_firma_id);
                $infor = $tipo_firma->tipo_firma;
                // dd($infor);
                if ($tipo_firma) {
                    switch ($infor) {
                        case 'ARCHIVO':
                            $contenedor = 1;
                            break;

                        case 'TOKEN':
                            $contenedor = 2;
                            break;

                        case 'NUBE':
                            $contenedor = 3;
                            break;

                        default:
                            $contenedor = 0;
                            break;
                    }
                } else {
                    // Maneja el caso en el que no se encuentre ningún registro con el precio_firma_id proporcionado
                }

                $solicitud_previa = SolicitudPrevia::create([
                    'solicitud_id' => null,
                    'email' => $request->email,
                    'identificacion' => $request->identificacion,
                    'telefono' => $request->telefono,
                    'usuario_corp' => $request->usuario_corp,
                    'entidad_id' => $request->entidad_id,
                    'estado' => true,
                    'contenedor' => $contenedor,
                    'requiere_facturacion' => $request->requiere_facturacion,
                    'numero_facturas' => $request->numero_facturas,
                    'requiere_firma' => $request->requiere_firma,
                    'vigenciafirma' => $tipo_firma->tiempo_validez,
                    'uanataca_id' => $request->uanataca_id,
                    'precio' => $request->precio,
                    'tipo_persona' => $request->tipo_persona,
                    'tipo_solicitud' => "NUEVA",
                    'datos_persona' => json_encode($request->datos_persona)
                ]);

                $destinatario = $request->email;
                Mail::to($destinatario)->send(new EmailSolicitudPrevia($solicitud_previa));

                return response()->json([
                    'status' => 201,
                    'message' => 'Solicitud previa creada correctamente.',
                    'data' => $solicitud_previa
                ], 201);
            }
        } catch (AuthorizationException $th) {
            return response()->json([
                'status' => $th->getCode(),
                'message' => 'No autorizado!.',
                'data' => $th->getMessage()
            ], 401);
        } catch (Exception $e) {
            return response()->json([
                'status' => $e->getCode(),
                'message' => 'Ocurrio un error!.',
                'data' => $e->getMessage()
            ], 400);
        }
    }

    public function listarSolicitudesEntidad(Request $request)
    {
        try {
            // $this->authorize('listar-solicitudes');
            $entidad = $request->input('entidad_id');
            $lst_solicitudes = SolicitudPrevia::orderBy('updated_at', 'asc')->get();
            return ($lst_solicitudes->isNotEmpty()) ?
                response()->json([
                    'status' => 200,
                    'message' => 'Lista de Solicitudes de Firma.',
                    'data' => $lst_solicitudes
                ], 200) :
                response()->json([
                    'status' => 200,
                    'message' => 'No hay Solicitudes de Firma.',
                    'data' => $lst_solicitudes
                ], 200);
        } catch (AuthorizationException $ae) {
            return response()->json([
                'status' => $ae->getCode(),
                'message' => 'No tiene permisos!.',
                'data' => $ae->getMessage()
            ], 401);
        } catch (Exception $e) {
            return response()->json([
                'status' => $e->getCode(),
                'message' => 'Ocurrió un error.',
                'data' => $e->getMessage()
            ], 400);
        }
    }

    public function reporteSolicitudesPrevias(Request $request)
    {
        try {
            // Obtener los datos de la solicitud previa y cargar las relaciones
            $entidad = $request->input('entidad_id');
        $usuario_corp = $request->input('usuario_corp');
        $fecha_inicio = $request->input('fecha_inicio');
        $fecha_fin = $request->input('fecha_fin');

        // Construir la consulta base
        $query = SolicitudPrevia::with('entidadComercial', 'usuario')
            ->whereBetween('created_at', [$fecha_inicio, $fecha_fin])
            ->orderBy('updated_at', 'asc');

        // Aplicar la condición de entidad_id si no es null
        if (!is_null($entidad)) {
            $query->where('entidad_id', $entidad);
        }

        // Aplicar la condición de usuario_corp si no es null
        if (!is_null($usuario_corp)) {
            $query->where('usuario_corp', $usuario_corp);
        }

        // Ejecutar la consulta
        $lst_solicitudes = $query->get();

            return ($lst_solicitudes->isNotEmpty()) ?
                response()->json([
                    'status' => 200,
                    'message' => 'Reporte de Solicitudes de Firma.',
                    'data' => $lst_solicitudes
                ], 200) :
                response()->json([
                    'status' => 200,
                    'message' => 'No hay Solicitudes de Firma para el reporte especificado.',
                    'data' => []
                ], 200);
        } catch (Exception $e) {
            return response()->json([
                'status' => $e->getCode(),
                'message' => 'Ocurrió un error al generar el reporte.',
                'data' => $e->getMessage()
            ], 400);
        }
    }

    public function listarSolicitudesPrevias(Request $request)
    {
        try {
            // $this->authorize('listar-solicitudes');
            $request->merge([
                'page' => $request->input('page', 0) + 1
            ]);
            $entidad = $request->input('entidad_id');
            $estado = $request->input('estado');
            $lst_solicitudes = SolicitudPrevia::with('entidadComercial.contacto', 'usuario') // Aquí cargamos la relación
            ->where('entidad_id', $entidad)
                ->when($estado, function ($query) use ($estado) {
                    return $query->where('estado', $estado);
                })
                ->orderBy('updated_at', 'desc')
                ->paginate($request->query('per_page') ?? 10);
            return ($lst_solicitudes != null) ?
                response()->json([
                    'status' => 200,
                    'message' => 'Lista de Solicitudes de Firma.',
                    'data' => $lst_solicitudes
                ], 200) :
                response()->json([
                    'status' => 200,
                    'message' => 'No hay Solicitudes de Firma.',
                    'data' => $lst_solicitudes
                ], 200);
        } catch (AuthorizationException $ae) {
            return response()->json([
                'status' => $ae->getCode(),
                'message' => 'No tiene permisos!.',
                'data' => $ae->getMessage()
            ], 401);
        } catch (Exception $e) {
            return response()->json([
                'status' => $e->getCode(),
                'message' => 'Ocurrio un error.',
                'data' => $e->getMessage()
            ], 400);
        }
    }

    public function listarSolicitudesKioscos(Request $request)
    {
        try {
            $request->merge([
                'page' => $request->input('page', 0) + 1
            ]);
            $entidad = $request->input('entidad_id');
            $estado = $request->input('estado');

            $lst_entidades = EntidadComercial::when($estado, function ($query) use ($estado) {
                return $query->where('estado', $estado);
            })
                ->where('nombre_completo', 'like', '%kiosko%') // Agregar esta condición
                ->orderBy('updated_at', 'desc')
                ->paginate($request->query('per_page') ?? 10);

            // Obtener los IDs de los registros que contienen "kiosko" en el campo nombre_completo
            $ids = $lst_entidades->pluck('id');
            $ids->push($entidad); // Agregar el contenido de $entidad al listado de IDs

            $lst_solicitudes = SolicitudPrevia::whereIn('entidad_id', $ids)
                ->when($estado, function ($query) use ($estado) {
                    return $query->where('estado', $estado);
                })
                ->orderBy('updated_at', 'desc')
                ->paginate($request->query('per_page') ?? 10);

            return response()->json([
                'status' => 200,
                'message' => 'No hay Solicitudes de Firma.',
                'data' => $lst_solicitudes
            ], 200);
        } catch (AuthorizationException $ae) {
            return response()->json([
                'status' => $ae->getCode(),
                'message' => 'No tiene permisos!.',
                'data' => $ae->getMessage()
            ], 401);
        } catch (Exception $e) {
            return response()->json([
                'status' => $e->getCode(),
                'message' => 'Ocurrio un error.',
                'data' => $e->getMessage()
            ], 400);
        }
    }


    public function actualizarSolicitud(Request $request, $id)
    {
        try {
            $solicitud = SolicitudPrevia::find($id);
            if ($solicitud != null) {
                $solicitud->update($request->all());
                return response()->json([
                    'status' => 200,
                    'message' => 'Solicitud de Firma previa actualizada correctamente.',
                    'data' => $solicitud
                ], 200);
            } else {
                return response()->json([
                    'status' => 200,
                    'message' => 'No se encontro la Solicitud de Firma previa indicada.',
                    'data' => null
                ], 200);
            }
        } catch (AuthorizationException $th) {
            return response()->json([
                'status' => $th->getCode(),
                'message' => 'No autorizado!.',
                'data' => $th->getMessage()
            ], 401);
        } catch (Exception $e) {
            return response()->json([
                'status' => $e->getCode(),
                'message' => 'Ocurrio un error!.',
                'data' => $e->getMessage()
            ], 400);
        }
    }

    public function sendEmail($id)
    {
        try {
            $solicitud = SolicitudPrevia::find($id);
            if ($solicitud != null) {
                $destinatario = $solicitud->email;
                Mail::to($destinatario)->send(new EmailSolicitudPrevia($solicitud));
                return response()->json([
                    'status' => 200,
                    'message' => 'Correo enviado',
                    'data' => $solicitud
                ], 200);
            } else {
                return response()->json([
                    'status' => 200,
                    'message' => 'No se encontro la Solicitud de Firma previa indicada.',
                    'data' => null
                ], 200);
            }
        } catch (AuthorizationException $th) {
            return response()->json([
                'status' => $th->getCode(),
                'message' => 'No autorizado!.',
                'data' => $th->getMessage()
            ], 401);
        } catch (Exception $e) {
            return response()->json([
                'status' => $e->getCode(),
                'message' => 'Ocurrio un error!.',
                'data' => $e->getMessage()
            ], 400);
        }
    }
}
